package gov.va.vamf.scheduling.varutility.security;

import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.dto.ClientAuthorizationRequest;
import com.agilex.healthcare.mobilehealthplatform.dto.HAOauth2Authentication;
import com.agilex.healthcare.mobilehealthplatform.dto.TokenValidationRequest;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import com.agilex.healthcare.utility.NullChecker;
import com.fasterxml.jackson.databind.AnnotationIntrospector;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.fasterxml.jackson.jaxrs.cfg.Annotations;
import com.fasterxml.jackson.jaxrs.json.JacksonJaxbJsonProvider;
import com.fasterxml.jackson.module.jaxb.JaxbAnnotationIntrospector;
import gov.va.vamf.scheduling.varutility.serializer.JSONJAXBContextResolver;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.provider.AuthorizationRequest;
import org.springframework.security.oauth2.provider.DefaultAuthorizationRequest;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.token.ResourceServerTokenServices;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.List;


public class CustomHATokenValidation implements ResourceServerTokenServices {

    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(CustomHATokenValidation.class);

    private String tokenValidationUri;

    public CustomHATokenValidation(String tokenValidationUri) {
        this.tokenValidationUri = tokenValidationUri;
    }

    @Override
    public OAuth2Authentication loadAuthentication(String accessToken) throws AuthenticationException {

        OAuth2Authentication oauth2Authentication = retrieveAndConstructOauth2Authentication(accessToken);
        return oauth2Authentication;
    }

    private OAuth2Authentication retrieveAndConstructOauth2Authentication(String accessToken) {
        OAuth2Authentication oauth2Authentication;

        TokenValidationRequest tokenValidationRequest = createTokenValidationRequest(accessToken);

        HAOauth2Authentication haOauth2Authentication = null;

        haOauth2Authentication = validateToken(tokenValidationRequest);

        oauth2Authentication = constructOauth2Authentication(haOauth2Authentication);

        return oauth2Authentication;
    }

    private OAuth2Authentication constructOauth2Authentication(HAOauth2Authentication haOauth2Authentication) {
        OAuth2Authentication oauth2Authentication;
        AuthorizationRequest authorizationRequest = createAuthorizationRequest(haOauth2Authentication);

        MhpUser mhpUser = haOauth2Authentication.getMhpUser();

        List<GrantedAuthority> authorities = createGrantedAuthorities(haOauth2Authentication.getAuthorities());

        AppUser appUser = new AppUser(haOauth2Authentication.getUserName(), "N/A", authorities, mhpUser);

        UsernamePasswordAuthenticationToken userAuthentication = new UsernamePasswordAuthenticationToken(appUser, "N/A", authorities);

        oauth2Authentication = new OAuth2Authentication(authorizationRequest, userAuthentication);
        return oauth2Authentication;
    }

    private List<GrantedAuthority> createGrantedAuthorities(List<String> authorities) {

        List<GrantedAuthority> grantedAuthorities = new ArrayList<GrantedAuthority>();

        if(NullChecker.isNullish(authorities)){
            return grantedAuthorities;
        }


        for (String authority : authorities) {
            grantedAuthorities.add(new SimpleGrantedAuthority(authority));
        }

        return grantedAuthorities;
    }

    private HAOauth2Authentication validateToken(TokenValidationRequest tokenValidationRequest) {
        JacksonJaxbJsonProvider provider = new JacksonJaxbJsonProvider(Annotations.JAXB);
        AnnotationIntrospector jaxbIntrospector = new JaxbAnnotationIntrospector(TypeFactory.defaultInstance());
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        mapper.configure(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY, true);
        mapper.setAnnotationIntrospector(jaxbIntrospector);
        provider.setMapper(mapper);
        Client restClient = ClientBuilder.newBuilder()
                .register(provider)
                .register(JacksonFeature.class)
                .register(JSONJAXBContextResolver.class)
                .build();
        WebTarget target = restClient.target(tokenValidationUri);

        HAOauth2Authentication haOauth2Authentication = null;
        haOauth2Authentication  = target.request(MediaType.APPLICATION_JSON)
                .post(Entity.entity(tokenValidationRequest, MediaType.APPLICATION_JSON), HAOauth2Authentication.class);

        return haOauth2Authentication;
    }

    private TokenValidationRequest createTokenValidationRequest(String accessToken) {
        TokenValidationRequest tokenValidationRequest = new TokenValidationRequest();
        tokenValidationRequest.setToken(accessToken);
        return tokenValidationRequest;
    }

    private AuthorizationRequest createAuthorizationRequest(HAOauth2Authentication haOauth2Authentication) {
        ClientAuthorizationRequest clientAuthorizationRequest = haOauth2Authentication.getAuthorizatioRequest();
        DefaultAuthorizationRequest authorizationRequest = new DefaultAuthorizationRequest(clientAuthorizationRequest.getClientId(),clientAuthorizationRequest.getScope());
        authorizationRequest.setApproved(clientAuthorizationRequest.isApproved());
        authorizationRequest.setRedirectUri(clientAuthorizationRequest.getRedirectUri());
        authorizationRequest.setResourceIds(clientAuthorizationRequest.getResourceIds());

        return authorizationRequest;
    }

    @Override
    public OAuth2AccessToken readAccessToken(String accessToken) {
        throw new UnsupportedOperationException();
    }
}